// module controls the ADC121 chip on the COLUTA tester board
// v1.0 jb/august/2018
//
//
//***************************************************************************
//
//  ADC121 control register definition:
//
//  ADC121_cntrl [4:0] SCLK speed
//                     clk is divided by factor ADC121_cntrl [4:0] + 10
//                     at clk=40MHz conversion speed will be 50kHz to 200KHz
//
//  ADC121_cntrl [5]   SP_1p2_A signal
//  ADC121_cntrl [6]   SP_1p2_B signal
//
//***************************************************************************

// v1.1 RXU/November/2018
// Make SP_1p2_{A&B} toggle at a really low frequency.  Take nothing from the user.

//
//
module ADC121
     (  
        //module state machine inputs
        input clk,
		  input rst,
		  input op_start,
		  input wire[63:0] ADC121_cntrl,
		  
		  //Dosimeter controls
		  output wire SP_1p2_A,
		  output wire SP_1p2_B,
		  
		  //ADC121 control signals
		  output reg CS_b,
		  output reg SCLK,
		  input      SDATA1,
		  input      SDATA2,
		  input      SDATA3,
		  input      SDATA4,
		  
		  //data_reg
		  output reg[11:0] Data1_reg,
		  output reg[11:0] Data2_reg,
		  output reg[11:0] Data3_reg,
		  output reg[11:0] Data4_reg,
        
		  //data_strobe
		  output reg data_strobe
		);
		


//variables			
reg       op_starts, op_start_reg, rst_reg, rst_flg;
reg [6:0] sclk_speed; 


//main body
always @(posedge clk)	
 begin	
//operation starts
          op_start_reg <= op_start;
          op_starts    <= ~op_start_reg & op_start;
//reset flag
          rst_reg <= rst;
          rst_flg <= ~rst_reg & rst;
//load SCLK speed
          if (rst_flg)        
           begin
		  
            sclk_speed <= {6{1'b0}};
            //SP_1p2_A   <= 1'b0;
            //SP_1p2_B   <= 1'b0;
			 
           end
          else if (op_starts) 
           begin
			
            sclk_speed <= ADC121_cntrl[4:0] + 5'b01010; //+10
            //SP_1p2_A   <= ADC121_cntrl[5];
            //SP_1p2_B   <= ADC121_cntrl[6];
			 
           end
 end


//SCLK signal is always running
reg [4:0] sclk_cnt;
reg [6:0] sclk_phase;

always @(posedge clk or posedge rst_flg)	
begin	
 if (rst_flg) 
  begin
  
   SCLK       <= 1'b1;
   sclk_phase <= {6{1'b0}};
	
  end
 else
  begin
  
  if (sclk_phase==sclk_speed) sclk_phase <= {6{1'b0}};
  else                        sclk_phase <= sclk_phase + 7'b0000001;
  
  if (sclk_phase<={1'b0, sclk_speed[6:1]}) SCLK <= 1'b0;
  else                                     SCLK <= 1'b1;
  
  end
end


//CS signal
always @(negedge SCLK or posedge rst_flg)	
begin	

if(rst_flg) sclk_cnt <= 5'b00000;

else
 begin
 
  if (sclk_cnt==5'b10100) sclk_cnt <= 5'b00001;
  else                    sclk_cnt <= sclk_cnt + 5'b00001;
  
 end
end

always @(posedge SCLK or posedge rst_flg)
begin

  if(rst_flg) CS_b <= 1'b1;
  
else
 begin
 
  if      (sclk_cnt==5'b10100) CS_b <= 1'b0;
  else if (sclk_cnt==5'b10000) CS_b <= 1'b1;
  
 end
end



//data registers for 4 ADC's
always @(negedge SCLK or posedge rst)
begin
 if (rst)
  begin
   Data1_reg <= 12'h001;
   Data2_reg <= 12'h002;
   Data3_reg <= 12'h003;
   Data4_reg <= 12'h004;	  
  end
 else if (CS_b==1'b0)
  begin
   Data1_reg <= {Data1_reg[10:0], SDATA1};
   Data2_reg <= {Data2_reg[10:0], SDATA2};	
   Data3_reg <= {Data3_reg[10:0], SDATA3};
   Data4_reg <= {Data4_reg[10:0], SDATA4};	
  end
end

//data_strobe; one pulse in clk (40MHz) domain
reg str_reg;
always @(posedge clk)
begin
 str_reg     <= sclk_cnt==5'b10001;
 data_strobe <= ~str_reg & (sclk_cnt==5'b10001);
end


// Really big clock divider.
reg [21:0] big_divider;
always @(posedge clk)
begin
	if (rst)		big_divider[21:0] <= {22{1'b0}};
	else			big_divider[21:0] <= big_divider[21:0] + 1'b1;
end
assign SP_1p2_A = (ADC121_cntrl[5]) ? big_divider[3] : big_divider[21];	// ADC121_cntrl[5] = 1: 5MHz output. ADC121_cntrl[5] = 0: approx 19 Hz.
assign SP_1p2_B = SP_1p2_A;

endmodule				